/*
  ==============================================================================

    SonicCrypt Seq
    Copyright (C) 2025 Sebastian Snkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#pragma once
#include <array>
#include <vector>
#include <cmath>
#include <JuceHeader.h> 

struct SequenceStep
{
    int noteOffset = 0;
    float velocity = 0.8f;
    bool active = false;
};

struct LayerSettings
{
    bool isActive = true;

    // Mute & Solo
    bool isMuted = false;
    bool isSoloed = false;

    int numSteps = 16;
    float swing = 0.0f;
    float gateLength = 0.6f;

    int scaleType = 1;      // 0=Chrom, 1=Min...
    int rootNote = 0;       // 0=C, 1=C#...
    int transpose = 0;      // Semitones
    int rateIndex = 2;      // 1/16 default
    int direction = 0;
    int octaveRange = 1;

    // Locks
    bool rateLocked = false;
    bool scaleLocked = false;
    bool rootLocked = false;
    bool transLocked = false;
    bool stepsLocked = false;
    bool dirLocked = false;
    bool octLocked = false;
    bool gateLocked = false;
    bool swingLocked = false;
};

class SequencerLayer
{
public:
    std::array<SequenceStep, 32> steps;
    LayerSettings settings;

    SequencerLayer() { clear(); }

    void clear()
    {
        for (auto& s : steps) { s.active = false; s.noteOffset = 0; s.velocity = 0.8f; }
        settings.isActive = true;
        settings.isMuted = false;
        settings.isSoloed = false;

        settings.numSteps = 16;
        settings.swing = 0.0f;
        settings.gateLength = 0.6f;
        settings.scaleType = 1;
        settings.rootNote = 0;
        settings.transpose = 0;
        settings.rateIndex = 2;
        settings.direction = 0;
        settings.octaveRange = 1;

        settings.rateLocked = false; settings.scaleLocked = false; settings.rootLocked = false;
        settings.transLocked = false;
        settings.stepsLocked = false; settings.dirLocked = false;
        settings.octLocked = false; settings.gateLocked = false;
        settings.swingLocked = false;
    }
};

class SequencerData
{
public:
    std::array<SequencerLayer, 3> layers;

    SequencerData()
    {
        layers[0].clear();
        layers[1].clear(); layers[1].settings.isActive = false;
        layers[2].clear(); layers[2].settings.isActive = false;
    }

    // XML EXPORT
    std::unique_ptr<juce::XmlElement> toXml()
    {
        auto xml = std::make_unique<juce::XmlElement>("SonicCryptSeq_Multi");

        for (int i = 0; i < 3; ++i)
        {
            auto* layerXml = new juce::XmlElement("Layer");
            layerXml->setAttribute("id", i);

            auto& s = layers[i].settings;
            layerXml->setAttribute("isActive", s.isActive);
            layerXml->setAttribute("isMuted", s.isMuted);
            layerXml->setAttribute("isSoloed", s.isSoloed);

            layerXml->setAttribute("numSteps", s.numSteps);
            layerXml->setAttribute("swing", s.swing);
            layerXml->setAttribute("gateLength", s.gateLength);
            layerXml->setAttribute("scaleType", s.scaleType);
            layerXml->setAttribute("rootNote", s.rootNote);
            layerXml->setAttribute("transpose", s.transpose);
            layerXml->setAttribute("rateIndex", s.rateIndex);
            layerXml->setAttribute("direction", s.direction);
            layerXml->setAttribute("octaveRange", s.octaveRange);

            layerXml->setAttribute("rateLocked", s.rateLocked);
            layerXml->setAttribute("scaleLocked", s.scaleLocked);
            layerXml->setAttribute("rootLocked", s.rootLocked);
            layerXml->setAttribute("transLocked", s.transLocked);
            layerXml->setAttribute("stepsLocked", s.stepsLocked);
            layerXml->setAttribute("dirLocked", s.dirLocked);
            layerXml->setAttribute("octLocked", s.octLocked);
            layerXml->setAttribute("gateLocked", s.gateLocked);
            layerXml->setAttribute("swingLocked", s.swingLocked);

            auto* stepsXml = new juce::XmlElement("Steps");
            for (int k = 0; k < 32; ++k)
            {
                auto* stepEl = new juce::XmlElement("Step");
                stepEl->setAttribute("idx", k);
                stepEl->setAttribute("act", layers[i].steps[k].active);
                stepEl->setAttribute("n", layers[i].steps[k].noteOffset);
                stepEl->setAttribute("v", layers[i].steps[k].velocity);
                stepsXml->addChildElement(stepEl);
            }
            layerXml->addChildElement(stepsXml);
            xml->addChildElement(layerXml);
        }
        return xml;
    }

    // XML IMPORT
    void fromXml(const juce::XmlElement& xml)
    {
        if (xml.getTagName() != "SonicCryptSeq_Multi") return;

        for (auto* layerXml : xml.getChildIterator())
        {
            if (layerXml->getTagName() == "Layer")
            {
                int id = layerXml->getIntAttribute("id", -1);
                if (id >= 0 && id < 3)
                {
                    auto& s = layers[id].settings;
                    s.isActive = layerXml->getBoolAttribute("isActive", true);
                    s.isMuted = layerXml->getBoolAttribute("isMuted", false);
                    s.isSoloed = layerXml->getBoolAttribute("isSoloed", false);

                    s.numSteps = layerXml->getIntAttribute("numSteps", 16);
                    s.swing = (float)layerXml->getDoubleAttribute("swing", 0.0);
                    s.gateLength = (float)layerXml->getDoubleAttribute("gateLength", 0.6);
                    s.scaleType = layerXml->getIntAttribute("scaleType", 1);
                    s.rootNote = layerXml->getIntAttribute("rootNote", 0);
                    s.transpose = layerXml->getIntAttribute("transpose", 0);
                    s.rateIndex = layerXml->getIntAttribute("rateIndex", 2);
                    s.direction = layerXml->getIntAttribute("direction", 0);
                    s.octaveRange = layerXml->getIntAttribute("octaveRange", 1);

                    s.rateLocked = layerXml->getBoolAttribute("rateLocked", false);
                    s.scaleLocked = layerXml->getBoolAttribute("scaleLocked", false);
                    s.rootLocked = layerXml->getBoolAttribute("rootLocked", false);
                    s.transLocked = layerXml->getBoolAttribute("transLocked", false);
                    s.stepsLocked = layerXml->getBoolAttribute("stepsLocked", false);
                    s.dirLocked = layerXml->getBoolAttribute("dirLocked", false);
                    s.octLocked = layerXml->getBoolAttribute("octLocked", false);
                    s.gateLocked = layerXml->getBoolAttribute("gateLocked", false);
                    s.swingLocked = layerXml->getBoolAttribute("swingLocked", false);

                    auto* stepsXml = layerXml->getChildByName("Steps");
                    if (stepsXml)
                    {
                        for (auto* stepEl : stepsXml->getChildIterator())
                        {
                            int k = stepEl->getIntAttribute("idx", -1);
                            if (k >= 0 && k < 32) {
                                layers[id].steps[k].active = stepEl->getBoolAttribute("act", false);
                                layers[id].steps[k].noteOffset = stepEl->getIntAttribute("n", 0);
                                layers[id].steps[k].velocity = (float)stepEl->getDoubleAttribute("v", 0.8);
                            }
                        }
                    }
                }
            }
        }
    }

    static std::vector<int> getScaleIntervals(int scaleType)
    {
        static const std::vector<std::vector<int>> scales = {
            {0,1,2,3,4,5,6,7,8,9,10,11}, // 0: Chromatic
            {0,2,3,5,7,8,10},            // 1: Minor
            {0,2,4,5,7,9,11},            // 2: Major
            {0,3,5,7,10},                // 3: Minor Pentatonic
            {0,4,7}                      // 4: Major Chord
        };
        if (scaleType < 0 || scaleType >= (int)scales.size()) return scales[0];
        return scales[scaleType];
    }

    static int quantizeNote(int inputOffset, int scaleType)
    {
        auto scale = getScaleIntervals(scaleType);
        int octave = inputOffset / 12;
        int noteInOctave = inputOffset % 12;
        if (noteInOctave < 0) { noteInOctave += 12; octave -= 1; }
        int bestMatch = scale[0];
        int minDistance = 100;
        for (int allowedNote : scale) {
            int dist = std::abs(noteInOctave - allowedNote);
            if (dist < minDistance) { minDistance = dist; bestMatch = allowedNote; }
        }
        return (octave * 12) + bestMatch;
    }
};